########Final analysis##########
#This script runs the empirical analysis in the paper. There are three parts that use different data sets/structures and outcome variables: onset of Pegida (survival analysis),
# particpants at pegida events within spells (OLS) and the temporary interruption of Pegida mobilization (conditional logit).

#load packages
library(ggplot2)
library(tidyverse)
library(survival)
library(plm)
library(texreg)
library(lmtest)
library(sjPlot)

#Clear environment
rm(list=ls())

#Load first data set. This contains a history of cities and up until they experience Pegida related mobilization for the first time. After that, they are excluded from the sample

#Load pegida protest spell data
load("data/pegida_history.Rda")

#### Table 2 #####
pegida_history$anti_demos_1month_bin <- ifelse(pegida_history$anti_demos_1month>0,1,0)

#Run survival models
fit1 <- coxph(Surv(start, stop, pegida_onset) ~  anti_demos_cumsum + anti_demos_1month_bin + cluster(city),
              data = pegida_history[pegida_history$city!="Dresden",], method = "efron")
summary(fit1)

fit2 <- coxph(Surv(start, stop, pegida_onset) ~  anti_demos_cumsum + anti_demos_1month_bin + log(einwohner)  + 
                I(primaersaldo/100) + beschä + geburtenrate + I(einkomsteuer/100)+ log(auslaender_total) + right_voteshare +
                ost + pegida_nationwide_lag + anti_pegida_nationwide_lag + cluster(city),
              data = pegida_history[pegida_history$city!="Dresden",], method = "efron")
summary(fit2)

fit3 <- coxph(Surv(start, stop, pegida_onset) ~  anti_demos_cumsum + anti_demos_1month_bin + log(einwohner)  + 
                I(primaersaldo/100) + beschä + geburtenrate + I(einkomsteuer/100) + log(auslaender_total) + right_voteshare  + cluster(city) + strata(bundesland),
              data = pegida_history[pegida_history$city!="Dresden",], method = "efron")
summary(fit3)

#Output to Latex
texreg::texreg(file = "output/table2.tex", list(fit1,fit2,fit3),
               label= "tab:coxresults",
               custom.note = "",
               stars = c(0.001, 0.01, 0.05, 0.1),
               symbol = "\\dagger",
               caption = "Relationshipe between counter-mobilization and RWP onset. Cox proportional hazard models. Significance levels: $^{***}$p$<$0.001; $^{**}$p$<$0.01; $^{*}$p$<$0.05; $^{dagger}p<0.1$.",
               booktabs=T,
               fontsize="footnotesize",
               custom.coef.names = c("Anti-Pegida demonstrations (cumulative)", "Anti-Pegida demonstration (prev. month)", "Inhabitants (log)", 
                                     "Public budget balance", "Employment rate", "Birth rate", "Income taxes", "Total immigrants (log)",
                                     "Vote share RW parties", "Eastern Germany", "Pegida protests (nationwide, t-1)", "Anti-Pegida protests (nationwide, t-1)"))


###Now we assess the effect of different types of counter-mobilization (size, actors, tactics) on the strength of PEGIDA
#Data: weekly protest spells, DV: Pegida participants

#### Table 3 #####

#Set options
options(digits = 3)

#Load pegida protest spell data
load("data/pegpad_waves.Rda")

#Pooled model (anti-Pegida participants)
plm_m1_a <- plm(I(participants_pegida_total_mean/100) ~ I(participants_anti_total_mean_lag/100) + I(anti_pegida_demos_lag>0)  + duration + I((duration^2)/10) +
                  time_trend + ost  + charlie_2month + cologne_2month + pegida_nationwide_lag + pegida_bundesland_lag + anti_pegida_nationwide_lag + anti_pegida_bundesland_lag, 
                data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0 ,], model = "pooling", effect = "individual", index = c("id", "stop"))
summary(plm_m1_a)

se_plm_m1_a <- coeftest(plm_m1_a, vcov = function(x) vcovHC(x, method = "arellano", type = "HC1", cluster="time"))
se_plm_m1_a

#within model with controls
plm_m1_b <- plm(I(participants_pegida_total_mean/100) ~  I(participants_anti_total_mean_lag/100)  + I(anti_pegida_demos_lag>0) + duration + I((duration^2)/10) +
                  charlie_2month + cologne_2month + pegida_nationwide_lag + pegida_bundesland_lag + anti_pegida_nationwide_lag + anti_pegida_bundesland_lag, 
                data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,], model = "within", effect = "individual", index = c("id", "stop"))
summary(plm_m1_b)

se_plm_m1_b <- coeftest(plm_m1_b, vcov = function(x) vcovHC(x, method = "arellano", type = "HC1", cluster="group"))
se_plm_m1_b

###

#Pooled model (anti-Pregida violence)
plm_m2_a <- plm(I(participants_pegida_total_mean/100) ~  part_violence_anti_pegida_max_lag + I(anti_pegida_demos_lag>0) + duration + I((duration^2)/10) +
                  time_trend + ost + charlie_2month + cologne_2month + pegida_nationwide_lag + pegida_bundesland_lag + anti_pegida_nationwide_lag + anti_pegida_bundesland_lag, 
                data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,], model = "pooling", effect = "individual",  index = c("id", "stop"))
summary(plm_m2_a)

se_plm_m2_a <- coeftest(plm_m2_a, vcov = function(x) vcovHC(x, method = "arellano", type = "HC1", cluster="group"))
se_plm_m2_a

#within model with controls
plm_m2_b <- plm(I(participants_pegida_total_mean/100) ~  part_violence_anti_pegida_max_lag  + I(anti_pegida_demos_lag>0) + duration + I((duration^2)/10) + charlie_2month + cologne_2month + pegida_nationwide_lag + pegida_bundesland_lag + anti_pegida_nationwide_lag + anti_pegida_bundesland_lag, 
                data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,], model = "within", effect = "individual", index = c("id", "stop"))
summary(plm_m2_b)

se_plm_m2_b <- coeftest(plm_m2_b, vcov = function(x) vcovHC(x, method = "arellano", type = "HC1", cluster="group"))
se_plm_m2_b

#within model with controls and both size and violence
plm_m2_c <- plm(I(participants_pegida_total_mean/100) ~  part_violence_anti_pegida_max_lag + I(participants_anti_total_mean_lag/100)  + I(anti_pegida_demos_lag>0) + duration + I((duration^2)/10) + charlie_2month + cologne_2month + pegida_nationwide_lag + pegida_bundesland_lag + anti_pegida_nationwide_lag + anti_pegida_bundesland_lag, 
                data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,], model = "within", effect = "individual", index = c("id", "stop"))
summary(plm_m2_c)

se_plm_m2_c <- coeftest(plm_m2_c, vcov = function(x) vcovHC(x, method = "arellano", type = "HC1", cluster="group"))
se_plm_m2_c

#Output to Latex
texreg(file = "output/table3.tex", list(plm_m1_a, plm_m1_b, plm_m2_a, plm_m2_b, plm_m2_c),
       label= "tab:panelresults",
       override.se = list(c(se_plm_m1_a[,2]), c(se_plm_m1_b[,2]), c(se_plm_m2_a[,2]), c(se_plm_m2_b[,2]), c(se_plm_m2_c[,2])),
       override.pvalues = list(c(se_plm_m1_a[,4]), c(se_plm_m1_b[,4]), c(se_plm_m2_a[,4]), c(se_plm_m2_b[,4]), c(se_plm_m2_c[,4])),
       custom.note = "",
       stars = c(0.001, 0.01, 0.05, 0.1),
       symbol = "\\dagger",
       caption = "Relationship between counter-mobilization and RWP intensity. Linear panel models with clustered standard errors. Significance levels: $^{***}$p$<$0.001; $^{**}$p$<$0.01; $^{*}$p$<$0.05; $^{dagger}p<0.1$.",
       booktabs=T,
       fontsize="footnotesize",
       custom.coef.names = c("Intercept", "Anti-Pegida participants (t-1)", "Anti-Pegida protest (t-1)",
                             "Spell duration", "Spell duration (squared)", "Linear time trend",
                             "Eastern Germany", "Charlie Hebdo", "Cologne incidents",
                             "Pegida protests (nationwide, t-1)", "Pegida protests (statewide, t-1)",
                             "Anti-Pegida protests (nationwide, t-1)", "Anti-Pegida protests (statewide, t-1)",
                             "Anti-Pegida violence (t-1)"),
       reorder.coef = c(2,14,8,9,10,11,12,13,3,4,5,6,7,1))

###
#Set options
options(digits = 3)

####Effect on disruption
#Data: weekly protest spells, DV: Pegida wave end

#### Table 4 #####

#Load pegida protest spell data
load("data/pegpad_waves.Rda")

#Conditional Logistic regression
#ANTI PARTICIPANTS
clogit_m1_a <- clogit(wave_end ~ I(participants_anti_total_mean/100) + duration + I((duration^2)/10) + time_trend + strata(city)  + cluster(city), data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,], method = "efron")
summary(clogit_m1_a)

#Logistic regression
clogit_m1_b <- clogit(wave_end ~ I(participants_anti_total_mean/100)  + charlie_2month + 
                        cologne_2month +  pegida_nationwide_lag + pegida_bundesland_lag + 
                        anti_pegida_nationwide_lag + anti_pegida_bundesland_lag +
                        time_trend + duration + I((duration^2)/10)  +
                        strata(city) + cluster(city),
                      data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,], method = "efron")
summary(clogit_m1_b)
####
#VIOLENCE

clogit_m2_a <- clogit(wave_end ~ part_violence_anti_pegida_max + duration + I((duration^2)/10) +
                        time_trend + strata(city) + cluster(city),
                      data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,], method = "efron")
summary(clogit_m2_a)

clogit_m2_b <- clogit(wave_end ~ part_violence_anti_pegida_max  + charlie_2month + 
                        cologne_2month +  pegida_nationwide_lag + pegida_bundesland_lag + 
                        anti_pegida_nationwide_lag + anti_pegida_bundesland_lag +
                        time_trend + duration + I((duration^2)/10)  +
                        strata(city) + cluster(city),
                      data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,],
                      method = "efron")
summary(clogit_m2_b)

clogit_m2_b <- clogit(wave_end ~ part_violence_anti_pegida_max  +
                        part_violence_pegida_max + charlie_2month + 
                        cologne_2month +  pegida_nationwide_lag + 
                        pegida_bundesland_lag +  anti_pegida_nationwide_lag +
                        anti_pegida_bundesland_lag +  time_trend + duration + I((duration^2)/10)  +
                        strata(city) + cluster(city),
                      data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,],
                      method = "efron")
summary(clogit_m2_b)

clogit_m2_c <- clogit(wave_end ~ part_violence_anti_pegida_max  +
                        part_violence_pegida_max + pol_engagement_pegida_med + charlie_2month + 
                        cologne_2month + pegida_nationwide_lag + pegida_bundesland_lag + 
                        anti_pegida_nationwide_lag + anti_pegida_bundesland_lag +
                        time_trend + duration + I((duration^2)/10)  +
                        strata(city) + cluster(city),
                      data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,],
                      method = "efron")
summary(clogit_m2_c)

# violence and participants
clogit_m2_d <- clogit(wave_end ~ part_violence_anti_pegida_max  + I(participants_anti_total_mean/100) +
                        part_violence_pegida_max + pol_engagement_pegida_med + charlie_2month + 
                        cologne_2month + pegida_nationwide_lag + pegida_bundesland_lag + 
                        anti_pegida_nationwide_lag + anti_pegida_bundesland_lag +
                        time_trend + duration + I((duration^2)/10)  +
                        strata(city) + cluster(city),
                      data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,],
                      method = "efron")
summary(clogit_m2_d)

# Output to Latex
texreg(file = "output/table4.tex", list(clogit_m1_a, clogit_m1_b, clogit_m2_a, clogit_m2_b, clogit_m2_c, clogit_m2_d),
       label= "tab:logitresults",
       custom.note = "",
       stars = c(0.001, 0.01, 0.05, 0.1),
       symbol = "\\dagger",
       caption = "Relationship between counter-mobilization and RWP demobilization. Conditional logistic regression models with clustered standard errors. Significance levels: $^{***}$p$<$0.001; $^{**}$p$<$0.01; $^{*}$p$<$0.05; $^{dagger}p<0.1$.",
       booktabs=T,
       fontsize="footnotesize",
       custom.coef.names = c("Anti-Pegida participants", "Spell duration", "Spell duration (squared)",
                             "Linear time trend", "Charlie Hebdo", "Cologne incidents",
                             "Pegida protests (nationwide)", "Pegida protests (statewide)",
                             "Anti-Pegida protests (nationwide)", "Anti-Pegida protests (statewide)",
                             "Anti-Pegida violence", "Pegida violence", "Police engagement"),
       reorder.coef = c(1,11,12,13,5,6,7,8,9,10,2,3,4))  

###

####### Figure 5 ########

# Recode variables
pegida_waves_final_rep$pegida_1000 <- pegida_waves_final_rep$participants_pegida_total_mean/1000
pegida_waves_final_rep$anti_pegida_1000 <- pegida_waves_final_rep$participants_anti_total_mean_lag/1000
pegida_waves_final_rep$duration_sq <- ((pegida_waves_final_rep$duration^2)/10)
pegida_waves_final_rep$anti_bin <- ifelse(pegida_waves_final_rep$anti_pegida_demos_lag>0,1,0)

# subset to non-missing observations
data <- pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0 & !is.na(pegida_waves_final_rep["anti_pegida_1000"]) &
                             !is.na(pegida_waves_final_rep["time_trend"])  &
                             !is.na(pegida_waves_final_rep["duration"]) & !is.na(pegida_waves_final_rep["pegida_1000"]) &
                             !is.na(pegida_waves_final_rep["anti_bin"]),]

# run model
plm_m1_marg <- plm(pegida_1000 ~ anti_bin + anti_pegida_1000  + part_violence_anti_pegida_max_lag + duration + duration_sq + time_trend + charlie_2month + 
                     cologne_2month + pegida_nationwide_lag + anti_pegida_nationwide_lag + pegida_bundesland_lag + anti_pegida_nationwide_lag + anti_pegida_bundesland_lag, 
                   data = data, model = "pooling", effect = "twoways", index = c("id", "stop"))
summary(plm_m1_marg)

# calculate effects
marginal_plot1 <- sjPlot::plot_model(plm_m1_marg, terms = "anti_pegida_1000", type = "pred", vcov.fun = vcovHC(plm_m1_marg, method = "arellano", type = "HC1", cluster="group"))

plot(marginal_plot1) + ggtitle("")  + theme_bw()  + ylab("Pegida participants (thousands)") + xlab("Anti-Pegida participants (thousands, t-1)") + scale_x_continuous(breaks = seq(0,40,5)) +
  geom_rug(pegida_waves_final_rep, mapping =  aes(x=anti_pegida_1000, y=pegida_1000),sides="b",position="jitter", alpha=0.5) + scale_y_continuous(limits =  c(-5,25), breaks = seq(0,25,5))

#Export to PDF
ggsave("output/Figure5.pdf", plot = last_plot(), 
       width = 7,
       height = 5,
       dpi = 600)

####### Figure 6 ########
pegida_waves_final_rep$duration_sq <- ((pegida_waves_final_rep$duration^2)/10)
pegida_waves_final_rep$anti_pegida_bin <-  ifelse(pegida_waves_final_rep$anti_pegida_demos>0,1,0) 
data_clogit <- pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0 &
                                    !is.na(pegida_waves_final_rep["part_violence_anti_pegida_max"]) &
                                    !is.na(pegida_waves_final_rep["part_violence_pegida_max"]),]

logistic_m1_marg<- glm(wave_end ~ part_violence_anti_pegida_max + part_violence_pegida_max +
                         anti_pegida_bin + charlie_2month  + pol_engagement_pegida_med +
                         cologne_2month + pegida_nationwide_lag + cologne_2month + pegida_bundesland_lag +
                         anti_pegida_nationwide_lag + anti_pegida_bundesland_lag + time_trend + duration +
                         duration_sq,
                       data = data_clogit,
                       family = "binomial")
summary(logistic_m1_marg)

dat <- margins::cplot(logistic_m1_marg, x = "part_violence_anti_pegida_max", se.type = "shade",
                      what = "prediction", type = "response") 

ggplot(dat, aes(x = xvals)) +
  geom_ribbon(aes(ymin = lower, ymax = upper), fill = "gray70", alpha = 0.55) +
  geom_line(aes(y = yvals)) +
  xlab("Anti-Pegida violence (t-1)") +
  ylab("Predicted value for Pegida termination") +
  ggtitle("") + 
  geom_rug(data_clogit, mapping =  aes(x=part_violence_anti_pegida_max, y=part_violence_anti_pegida_max), 
           sides="b",position="jitter", alpha=0.5)  + ylim(c(-0.05,0.35)) +
  theme_bw()

#Export to PDF
ggsave("output/Figure6.pdf", plot = last_plot(), 
       width = 7,
       height = 5,
       dpi = 600)

########ROBUSTNESS CHECKS######

##### Figure A.2 #####

# Recode variables for plot_model to work
#within model with both variables
pegida_waves_subset <- pegida_waves_final_rep %>% filter(pegida_demos > 0) %>% 
  mutate(part_pegida_mean_1000 = participants_pegida_total_mean/1000,
         part_anti_pegida_1000 = participants_anti_total_mean_lag/1000,
         anti_yes_no = ifelse(anti_pegida_demos_lag > 0, 1, 0),
         duration_sq = (duration^2)/10,
         east_factor = factor(ost)) %>% 
  drop_na(part_violence_anti_pegida_max_lag,
          participants_anti_total_mean_lag,
          part_pegida_mean_1000,
          part_anti_pegida_1000,
          anti_pegida_demos_lag,
          duration,
          charlie_2month,
          pegida_nationwide_lag,
          pegida_bundesland_lag,
          anti_pegida_bundesland_lag,
          anti_pegida_nationwide_lag)

# run model with interaction effect
plm_m3_c <- plm(part_pegida_mean_1000 ~  part_anti_pegida_1000*part_violence_anti_pegida_max_lag   +
                  anti_yes_no + duration + duration_sq + charlie_2month + cologne_2month + pegida_nationwide_lag + 
                  pegida_bundesland_lag + anti_pegida_nationwide_lag + anti_pegida_bundesland_lag, 
                data = pegida_waves_subset, model = "pooling", effect = "twoway", index = c("id", "stop"))
summary(plm_m3_c)

se_plm_m3_c <- coeftest(plm_m3_c, vcov = function(x) vcovHC(x, method = "arellano", type = "HC1", cluster="group"))
se_plm_m3_c

# Plot interaction effects
sjPlot::plot_model(plm_m3_c, type = "int", mdrt.values = "minmax", 
                   vcov.fun = function(x) vcovHC(x, method = "arellano", type = "HC1", cluster="group")) +
  ggtitle("") +
  ylab("Pegida participants (thousands)") +
  xlab("Anti-Pegida participants (thousands, t-1)") +
  scale_color_manual("Anti-Pegida violence", breaks = c(0,4),
                     labels = c("no violence", "physical attacks"),
                     values = c("grey30", "grey70")) +
  scale_fill_manual("Anti-Pegida violence", breaks = c(0,4),
                    labels = c("no violence", "physical attacks"),
                    values = c("grey30", "grey70")) +
  theme_bw()


#Export to PDF
ggsave("output/FigureA2.pdf", plot = last_plot(), 
       width = 8,
       height = 6,
       limitsize = TRUE,
       dpi = 600)

####

##### Figure A.3 #####

# Recode variables for plot_model to work
pegida_waves_subset <- pegida_waves_final_rep %>% filter(pegida_demos > 0) %>% 
  mutate(part_pegida_mean_1000 = participants_pegida_total_mean/1000,
         part_anti_pegida_1000 = participants_anti_total_mean/1000,
         anti_yes_no = ifelse(anti_pegida_demos_lag > 0, 1, 0),
         duration_sq = (duration^2)/10) %>% 
  drop_na(part_violence_anti_pegida_max_lag,
          participants_anti_total_mean_lag,
          part_pegida_mean_1000,
          part_anti_pegida_1000,
          anti_pegida_demos_lag,
          duration,
          charlie_2month,
          pegida_nationwide_lag,
          pegida_bundesland_lag,
          anti_pegida_bundesland_lag,
          anti_pegida_nationwide_lag)


# violence and participants
clogit_m2_e <- clogit(wave_end ~ part_anti_pegida_1000*part_violence_anti_pegida_max +
                        part_violence_pegida_max + pol_engagement_pegida_med + charlie_2month + 
                        cologne_2month + pegida_nationwide_lag + pegida_bundesland_lag + 
                        anti_pegida_nationwide_lag + anti_pegida_bundesland_lag +
                        time_trend + duration + duration_sq + cluster(city),
                      data = pegida_waves_subset,
                      method = "efron")
summary(clogit_m2_e)


sjPlot::plot_model(clogit_m2_e, type = "int", mdrt.values = "minmax") +
  ggtitle("") + ylim(0,10) +
  xlab("Anti-Pegida participants (thousands, t-1)") +
  scale_color_manual("Anti-Pegida violence", breaks = c(0,4),
                     labels = c("no violence", "physical attacks"),
                     values = c("grey30", "grey70")) +
  scale_fill_manual("Anti-Pegida violence", breaks = c(0,4),
                    labels = c("no violence", "physical attacks"),
                    values = c("grey30", "grey70")) +
  theme_bw()


#Export to PDF
ggsave("output/FigureA3.pdf", plot = last_plot(), 
       width = 8,
       height = 6,
       limitsize = F,
       dpi = 600)


#### Table A4 #####
# A1, including sum of anti-Pegida demos
fitA1_1 <- coxph(Surv(start, stop, pegida_onset) ~  anti_demos_cumsum + anti_demos_1month + log(einwohner)  +
                   I(primaersaldo/100) + beschä + geburtenrate + I(einkomsteuer/100) + log(auslaender_total) +
                   right_voteshare + ost + cluster(city),
                 data = pegida_history[pegida_history$city!="Dresden",], method = "efron")
summary(fitA1_1)

# A2, run clogit
fitA1_2 <- clogit(pegida_onset ~  anti_demos_cumsum + anti_demos_1month_bin + log(einwohner)  +
                    I(primaersaldo/100) + beschä + geburtenrate + I(einkomsteuer/100) + log(auslaender_total) +
                    right_voteshare + ost + cluster(city) + stop + I(stop^2) + pegida_nationwide_lag +
                    anti_pegida_nationwide_lag,
                  data = pegida_history[pegida_history$city!="Dresden",], method = "efron")
summary(fitA1_2)

# A3, rare event logit
library(brglm)
brglm_A1_3 <- brglm(pegida_onset ~ anti_demos_cumsum + anti_demos_1month_bin + log(einwohner)  +
                      I(primaersaldo/100) + beschä + geburtenrate + I(einkomsteuer/100) + log(auslaender_total) +
                      right_voteshare + ost  + stop + I(stop^2) + pegida_nationwide_lag +
                      anti_pegida_nationwide_lag,
                    family = binomial(logit),
                    data = pegida_history[pegida_history$city!="Dresden",],
                    method = "brglm.fit", control.brglm = brglm.control(br.maxit = 1000), pl=T)
summary(brglm_A1_3)

#Export to Table A1###
texreg::texreg(file = "output/tableA4.tex", list(fitA1_1,fitA1_2,brglm_A1_3),
               label= "tab:A4",
               custom.note = "",
               stars = c(0.001, 0.01, 0.05, 0.1),
               symbol = "\\dagger",
               caption = "Robustness checks. Relationship between counter-mobilization and RWP onset. Models with the absolute number of counter-demonstrations in the previous month (M1), binary logit (M2) and rare event logistic regression (M3) with time trends and controls for nationwide protest events. Significance levels: $^{***}$p$<$0.001; $^{**}$p$<$0.01; $^{*}$p$<$0.05; $^{dagger}p<0.1$.",
               booktabs=T,
               fontsize="footnotesize",
               custom.coef.names = c("Anti-Pegida demonstrations (cumulative)",
                                     "Anti-Pegida demonstrations (#, prev. month)", "Inhabitants (log)", 
                                     "Public budget balance", "Employment rate", "Birth rate", "Income taxes",
                                     "Total immigrants (log)", "Vote share RW parties", "Eastern Germany", 
                                     "Anti-Pegida demonstration (prev. month)",
                                     "Time trend", "Time trend (squared)", "Pegida protests (nationwide, t-1)",
                                     "Anti-Pegida protests (nationwide, t-1)", "Intercept"))

#### Table A5 #####

# Run FE models with alternative specifications of the indep variable: MIN participants
plm_A2_1 <- plm(I(participants_pegida_total_min/100)  ~  I(participants_anti_total_min_lag/100)  +
                  I(anti_pegida_demos_lag>0) + duration + I((duration^2)/10) + charlie_2month +
                  cologne_2month + pegida_nationwide_lag + anti_pegida_nationwide_lag, 
                data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,],
                model = "within", effect = "individual", index = c("id"))
summary(plm_A2_1)

se_plm_A2_1 <- coeftest(plm_A2_1, vcov = function(x) vcovHC(x, method = "arellano", type = "HC1", cluster="group"))
se_plm_A2_1

# Run FE models with alternative specifications of the indep variable: MAX participants
plm_A2_2 <- plm(I(participants_pegida_total_max/100)  ~  I(participants_anti_total_max_lag/100)  +
                  I(anti_pegida_demos_lag>0) + duration + I((duration^2)/10) + charlie_2month +
                  cologne_2month + pegida_nationwide_lag + anti_pegida_nationwide_lag, 
                data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,],
                model = "within", effect = "individual", index = c("id"))
summary(plm_A2_2)

se_plm_A2_2 <- coeftest(plm_A2_2, vcov = function(x) vcovHC(x, method = "arellano", type = "HC1", cluster="group"))
se_plm_A2_2

# Run FE models with alternative specifications of the indep variable: LOG
plm_A2_3 <- plm(I(log(participants_pegida_total_mean))  ~  I(log(participants_anti_total_mean_lag+1))  +
                  I(anti_pegida_demos_lag>0) + duration + I((duration^2)/10) + charlie_2month +
                  cologne_2month + pegida_nationwide_lag + anti_pegida_nationwide_lag, 
                data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,],
                model = "within", effect = "individual", index = c("id"))
summary(plm_A2_3)

se_plm_A2_3 <- coeftest(plm_A2_3, vcov = function(x) vcovHC(x, method = "arellano", type = "HC1", cluster="group"))
se_plm_A2_3

# Run FE models with alternative specifications of the indep variable: MED violence
plm_A2_4 <- plm(I(participants_pegida_total_mean/100) ~  part_violence_anti_pegida_med_lag  +
                  I(anti_pegida_demos_lag>0) + duration + I((duration^2)/10) + charlie_2month +
                  cologne_2month + pegida_nationwide_lag + anti_pegida_nationwide_lag, 
                data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,], model = "within", effect = "individual", index = c("id"))
summary(plm_A2_4)

se_plm_A2_4 <- coeftest(plm_A2_4, vcov = function(x) vcovHC(x, method = "arellano", type = "HC1", cluster="group"))

# Run FE models with alternative specifications of the indep variable: MED violence
plm_A2_5 <- plm(I(participants_pegida_total_mean/100) ~  part_violence_anti_pegida_med_lag  + 
                  part_violence_pegida_med_lag +
                  I(anti_pegida_demos_lag>0) + duration + I((duration^2)/10) + charlie_2month +
                  cologne_2month + pegida_nationwide_lag + anti_pegida_nationwide_lag, 
                data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,], model = "within", effect = "individual", index = c("id"))
summary(plm_A2_5)

se_plm_A2_5 <- coeftest(plm_A2_5, vcov = function(x) vcovHC(x, method = "arellano", type = "HC1", cluster="group"))


# Export to Table
texreg(file = "output/tableA5.tex", list(plm_A2_1, plm_A2_2, plm_A2_3, plm_A2_4, plm_A2_5),
       label= "tab:A5",
       override.se = list(c(se_plm_A2_1[,2]), c(se_plm_A2_2[,2]), c(se_plm_A2_3[,2]), c(se_plm_A2_4[,2]), c(se_plm_A2_5[,2])),
       override.pvalues = list(c(se_plm_A2_1[,4]), c(se_plm_A2_2[,4]), c(se_plm_A2_3[,4]), c(se_plm_A2_4[,4]), c(se_plm_A2_5[,4])),
       custom.note = "",
       stars = c(0.001, 0.01, 0.05, 0.1),
       symbol = "\\dagger",
       caption = "Robustness checks. Relationship between counter-mobilization and RWP size. Linear panel models with clustered standard errors. Significance levels: $^{***}$p$<$0.001; $^{**}$p$<$0.01; $^{*}$p$<$0.05; $^{dagger}p<0.1$.",
       booktabs=T,
       fontsize="footnotesize",
       custom.coef.names = c("Anti-Pegida participants (min, t-1)", "Anti-Pegida protest (t-1)",
                             "Spell duration", "Spell duration (squared)",
                             "Charlie Hebdo", "Cologne incidents",
                             "Pegida protests (nationwide, t-1)", "Anti-Pegida protests (nationwide, t-1)",
                             "Anti-Pegida participants (max, t-1)",
                             "Anti-Pegida participants (log, t-1)",
                             "Anti-Pegida violence (median, t-1)", 
                             "Pegida violence (t-1)"),
       reorder.coef = c(1,9,10,11,12,5,6,7,8,2,3,4))

#### Table A6 #####

#### Clogit
#participant numbers MIN

clogit_A3_1 <- clogit(wave_end ~ I(participants_anti_total_min/100) + charlie_2month + 
                        cologne_2month + pegida_nationwide + anti_pegida_nationwide +
                        time_trend + duration + I((duration^2)/10)  +
                        strata(city) + cluster(city),
                      data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,],
                      method = "efron")
summary(clogit_A3_1)

#participant numbers MAX
clogit_A3_2 <- clogit(wave_end ~ I(participants_anti_total_max/100) + charlie_2month + 
                        cologne_2month + pegida_nationwide + anti_pegida_nationwide +
                        time_trend + duration + I((duration^2)/10)  +
                        strata(city) + cluster(city),
                      data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,],
                      method = "efron")
summary(clogit_A3_2)

#including pegida mobilization
clogit_A3_3 <- clogit(wave_end ~ I(participants_anti_total_mean/100) + 
                        I(participants_pegida_total_mean/100) + charlie_2month + 
                        cologne_2month + pegida_nationwide + anti_pegida_nationwide  +
                        time_trend + duration + I((duration^2)/10)  +
                        strata(city) + cluster(city),
                      data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,],
                      method = "efron")
summary(clogit_A3_3)

#medium anti violence
clogit_A3_4 <- clogit(wave_end ~ part_violence_anti_pegida_med + part_violence_pegida_med +
                        pol_engagement_pegida_med +
                        charlie_2month + 
                        cologne_2month + pegida_nationwide + anti_pegida_nationwide +
                        time_trend + duration + I((duration^2)/10)  +
                        strata(city) + cluster(city),
                      data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,],
                      method = "efron")
summary(clogit_A3_4)

#all organizations
clogit_A3_5 <- clogit(wave_end ~ civ_soc_anti + repre_pol + church_anti + left_assoc_anti +
                        party_anti  +
                        charlie_2month + 
                        cologne_2month + pegida_nationwide + anti_pegida_nationwide +
                        time_trend + duration + I((duration^2)/10)  +
                        strata(city) + cluster(city),
                      data = pegida_waves_final_rep[pegida_waves_final_rep$pegida_demos>0,],
                      method = "efron")
summary(clogit_A3_5)

##Different thresholds for disruption##
# 2 weeks


#Load pegida protest spell data
load("data/pegpad_waves_2weeks.Rda")

#Logistic regression
clogit_m1_b_A1 <- clogit(wave_end ~ I(participants_anti_total_mean/100)  + charlie_2month + 
                           cologne_2month + pegida_nationwide + anti_pegida_nationwide + time_trend +
                           duration + I((duration^2)/10)  + strata(city) + cluster(city),
                         data = pegida_waves_final_rep_2w[pegida_waves_final_rep_2w$pegida_demos>0,], method = "efron")
summary(clogit_m1_b_A1)

clogit_m2_b_A2 <- clogit(wave_end ~ part_violence_anti_pegida_max  +
                           part_violence_pegida_max + charlie_2month + 
                           cologne_2month + pegida_nationwide + anti_pegida_nationwide +
                           time_trend + duration + I((duration^2)/10)  +
                           strata(city) + cluster(city),
                         data = pegida_waves_final_rep_2w[pegida_waves_final_rep_2w$pegida_demos>0,],
                         method = "efron")
summary(clogit_m2_b_A2)


#Load pegida protest spell data
load("data/pegpad_waves_8weeks.Rda")

#Logistic regression
clogit_m1_b_A3 <- clogit(wave_end ~ I(participants_anti_total_mean/100)  + charlie_2month + 
                           cologne_2month + pegida_nationwide + anti_pegida_nationwide + time_trend +
                           duration + I((duration^2)/10)  + strata(city) + cluster(city),
                         data = pegida_waves_final_rep_8w[pegida_waves_final_rep_8w$pegida_demos>0,], method = "efron")
summary(clogit_m1_b_A3)

clogit_m2_b_A4 <- clogit(wave_end ~ part_violence_anti_pegida_max  +
                           part_violence_pegida_max + charlie_2month + 
                           cologne_2month + pegida_nationwide + anti_pegida_nationwide +
                           time_trend + duration + I((duration^2)/10)  +
                           strata(city) + cluster(city),
                         data = pegida_waves_final_rep_8w[pegida_waves_final_rep_8w$pegida_demos>0,],
                         method = "efron")
summary(clogit_m2_b_A4)

texreg(file = "output/tableA6.tex", list(clogit_A3_1, clogit_A3_2, clogit_A3_3, clogit_A3_4, clogit_A3_5, clogit_m2_b_A2,
            clogit_m2_b_A4),
       label= "tab:A6",
       custom.note = "",
       stars = c(0.001, 0.01, 0.05, 0.1),
       symbol = "\\dagger",
       caption = "Robustness checks. Relationship between counter-mobilization and RWP demobilization Conditional logistic regression models with clustered standard errors. Significance levels: $^{***}$p$<$0.001; $^{**}$p$<$0.01; $^{*}$p$<$0.05; $^{dagger}p<0.1$.",
       booktabs=T,
       fontsize="footnotesize",
       custom.coef.names = c("Anti-Pegida participants (min)", "Charlie Hebdo",
                             "Cologne incidents", "Pegida protests (nationwide)",
                             "Anti-Pegida protests (nationwide)", 
                             "Linear time trend",
                             "Duration", "Duration (squared)", "Anti-Pegida participants (max)",
                             "Anti-Pegida participants (mean)", "Pegida participants (mean)",
                             "Anti-Pegida violence (median)", "Pegida violence (median)",
                             "Police engagement",
                             "Civil society actors", "Political representatives",
                             "Churches", "Left-wing associations", "Political parties",
                             "Anti-Pegida violence" , "Pegida violence"),
       reorder.coef = c(1,9,10,11,12,13,14,20,21,15,16,17,18,19,2,3,4,5,6,7,8)) 






